# CA Task Manager

## Overview

This application is a **Streamlit** based task and reminder management system for an Indian Chartered Accountant (CA) office.  It provides a simple yet powerful way to manage employees, tasks, task assignments, email reminders and management information system (MIS) reports.  Persistence is handled via **TinyDB** (JSON‑backed), and email reminders are sent through the **Gmail API**.

### Key Features

* **Authentication & RBAC** – Basic login screen with hashed passwords (bcrypt).  Users are assigned one of three roles: Admin, Manager, or Staff.  Different pages and actions are available based on the role.
* **Employee Management** – Create, read, update and deactivate employees.  Each employee has a unique `emp_code`, contact details and a date of joining.  Only active employees with valid email addresses will receive reminders.
* **Task Management** – Maintain a catalogue of regulatory and client tasks including codes, due dates, recurrence, priority and notes.  Tasks cannot be created with past due dates.  Recurring tasks automatically spawn the next instance upon completion.
* **Task Assignment** – Assign tasks to one or more employees.  Track status, percentage completion, remarks and actual completion date.  Bulk assignment and update is supported.
* **Email Reminders** – Remind assigned employees of upcoming due dates.  A daily scheduler runs at **09:00 IST** using APScheduler.  The look‑ahead window can be configured in the settings page (default 7 days).  A “Send Now” button triggers immediate reminders.  Email logs are stored for auditing.
* **MIS Reports** – Management dashboards built using **Pandas** and **Plotly** summarise workload, completion ratios, delays and distribution of tasks across employees.  Reports can be exported to CSV/Excel.
* **Compliance Calendar** – A calendar view highlights recurring statutory tasks by month, helping managers plan ahead.
* **Backup & Export** – All data lives under the `data/` folder as JSON files.  Creating a backup is as simple as copying these files.  MIS data can be exported.

## Project Structure

```
ca_task_manager/
├── app.py                # Streamlit entry point
├── scheduler_runner.py   # CLI for running the reminder scheduler
├── models/               # Pydantic models and dataclasses
├── repositories/         # TinyDB repositories
├── services/             # Business logic and operations
├── ui_pages/             # Streamlit page definitions
├── utils/                # Helper utilities (hashing, email, timezone)
├── data/                 # JSON storage (populated at runtime)
├── .env.example         # Example environment variables
├── .streamlit/
│   └── secrets.toml      # Example secrets for deployment
├── requirements.txt      # Python dependencies
├── seed_data.py          # Script to populate sample data
└── README.md             # This file
```

## Getting Started

1. **Clone the repository and install dependencies**:

   ```bash
   python -m venv venv
   source venv/bin/activate
   pip install -r requirements.txt
   ```

2. **Create an `.env` file** in the project root by copying `.env.example` and fill in the environment variables.  At a minimum you need to define `EMAIL_LOOKAHEAD_DAYS` and `TIMEZONE`.  When deploying to Streamlit Cloud, put secrets in `.streamlit/secrets.toml` instead.

3. **Configure Gmail API**:

   * Create a project on the [Google Cloud Console](https://console.cloud.google.com/).
   * Enable the **Gmail API** for that project.
   * Create OAuth credentials of type “Desktop App”.
   * Download the `credentials.json` file and place it in the project root.
   * When first running the app, you will be prompted to authenticate and consent; a `token.json` will be created and stored in `data/`.

   For more details see the [Gmail Python Quickstart](https://developers.google.com/gmail/api/quickstart/python).

4. **Seed sample data** (optional):

   ```bash
   python seed_data.py
   ```

5. **Run the application**:

   ```bash
   streamlit run app.py
   ```

6. **Run the scheduler as a CLI**:

   ```bash
   python scheduler_runner.py --send-now
   ```

## Environment Variables

The application uses environment variables loaded from `.env` or `st.secrets`.  The following are supported:

| Variable              | Description                                               | Example |
|---------------------- |-----------------------------------------------------------|---------|
| `TIMEZONE`            | IANA timezone string used for scheduling (default IST)    | `Asia/Kolkata` |
| `EMAIL_LOOKAHEAD_DAYS`| Number of days ahead to include in reminders (integer)    | `7` |
| `GMAIL_USER`          | Email address used to send reminders                     | `yourname@example.com` |

Secrets such as client IDs and refresh tokens are stored in `secrets.toml`.  Never commit real credentials to version control.

## Data Persistence

All business entities (users, employees, tasks, assignments, email logs, settings) are persisted in TinyDB JSON files under the `data/` directory.  The repository layer encapsulates CRUD operations and enforces unique constraints on fields like employee code and task code.  Feel free to back up this folder regularly.

## Security

* Passwords are hashed with **bcrypt** via Passlib and never stored in plaintext.
* Only active employees with valid email addresses receive reminders.
* Minimal Role Based Access Control (RBAC) is provided: **Admin** can manage everything, **Manager** can assign tasks and view reports, **Staff** can view and update their own assignments.
* Sensitive secrets must be stored in environment variables or Streamlit secrets; never hard‑code them.

## Contributing

Pull requests and suggestions are welcome!  Please open an issue if you spot a bug or have a feature request.